/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package viewer;

import java.awt.BorderLayout;
import java.awt.Color;
import java.net.URISyntaxException;

import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;

import com.inet.report.EngineRenderData;
import com.inet.viewer.RenderData;
import com.inet.viewer.ReportView;
import com.inet.viewer.SwingReportViewer;

/**
 * This sample demonstrates how you can customize the viewer's status bar, remove it, or even replace it with your own
 * component.
 */
public class CustomStatusBar extends JFrame {

    private RenderData        renderConnection; // This is our main render data connection - the source of our raw report data coming from our report server

    private SwingReportViewer viewer;           // Our top-level viewer object

    /**
     * Customize the viewer's status bar.
     */
    public CustomStatusBar() {
        super( "i-net Clear Reports - Custom Status Bar" );
        // to initialize we first create a top level ReportViewer:
        viewer = new SwingReportViewer();

        String reportLocation = "";
        try {
            reportLocation = getClass().getResource( "../sample.rpt" ).toURI().toString();
        } catch( URISyntaxException e ) {
            // Nothing to do
        }

        // then initialize the render data connection.
        renderConnection = new EngineRenderData( "report=" + reportLocation );

        // you will most likely have a report server already, so you can use the URL: http://server:port/report.rpt
        // renderConnection = new URLRenderData( "http://server:port/?report=file:c:/report1.rpt" );

        // addNewReportView causes a new report view to be created using the given connection as its data source, and then
        // for this newly created report to be added to the viewer.
        // We store this report view to start working with its status bar
        ReportView view = viewer.addNewReportView( renderConnection );

        // This would remove the original status bar
        //
        // view.extractStatusBar();

        // Now we create a new status bar component
        JPanel ourStatusBar = new JPanel();
        ourStatusBar.setBackground( new Color( 0.7f, 0.4f, 0.4f ) );
        ourStatusBar.add( new JLabel( "My very own status bar!" ) );

        // Here we replace the status bar with our own component. Note that this is only in a GUI
        // sense, the StatusBar above is still receiving status messages, our component will get no
        // status messages.
        view.replaceStatusBar( ourStatusBar );

        // add the viewer to the target window
        getContentPane().add( BorderLayout.CENTER, viewer );

        pack();
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        (new CustomStatusBar()).setVisible( true );
    }
}
